package dev.bnjc.bglib;

import dev.bnjc.bglib.exceptions.BGIParseException;
import dev.bnjc.bglib.exceptions.ErrorCode;
import dev.bnjc.bglib.utils.ByteParser;
import java.nio.ByteBuffer;
import java.nio.ByteOrder;
import java.util.HashMap;
import java.util.Map;
import java.util.Optional;
import net.minecraft.class_1799;
import net.minecraft.class_2487;
import net.minecraft.class_2520;
import net.minecraft.class_9279;
import net.minecraft.class_9334;

/**
 * A parser to parse the custom Blockgame Item (BGI) tag found in the custom data component.
 *
 * <pre>
 *   BGIResult result = BGIParser.parse(itemStack);
 *   result.ifSuccess((data) -> {
 *     String itemId = data.getString(BGIField.ITEM_ID);
 *   });
 * </pre>
 *
 * @author Jack Grzechowiak
 * @since 0.1.2
 */
public final class BGIParser {
  /**
   * The name of the Blockgame Item byte array tag used within an {@linkplain class_1799} custom data component
   */
  public static final String BGI_TAG = "bgi";

  private final ByteBuffer buffer;
  private final byte[] originalData;

  private BGIParser(byte[] data) {
    this.originalData = data;
    this.buffer = ByteBuffer.wrap(data);
    this.buffer.order(ByteOrder.BIG_ENDIAN);
  }

  /**
   * Parses the specified byte array into a {@link BGIParseResult} object.
   *
   * @param data Byte array from NBT data
   * @return a {@link BGIParseResult} corresponding to the specified byte array
   * @since 0.1.1
   */
  public static BGIParseResult<BGIData> parse(byte[] data) {
    var parser = new BGIParser(data);
    return parser.parse();
  }

  /**
   * Parses the specified item stack's custom data component into a {@link BGIParseResult} object. If the stack
   * does not have the "bgi" tag in its custom data, then an error will be returned.
   *
   * @param itemStack Item stack to find the tag in
   * @return a {@link BGIParseResult} corresponding to the specified item stack
   */
  public static BGIParseResult<BGIData> parse(class_1799 itemStack) {
    class_2487 stackNbt = itemStack.method_57825(class_9334.field_49628, class_9279.field_49302).method_57461();

    if (stackNbt != null && stackNbt.method_10573(BGI_TAG, class_2520.field_33257)) {
      return BGIParser.parse(stackNbt.method_10547(BGI_TAG));
    }

    return BGIParseResult.error(ErrorCode.MISSING_TAG);
  }

  /**
   * Attempts to determine the matching {@link BGIField} from the given hash code. This is most useful for debugging
   * purposes as it will loop over all the fields to match the hash.
   *
   * @param hashCode The field name hash code to match
   * @return if present, the matching {@link BGIField}
   */
  public static Optional<BGIField> getFieldFromHash(int hashCode) {
    for (var field : BGIField.values()) {
      if (field.name().hashCode() == hashCode) {
        return Optional.of(field);
      }
    }
    return Optional.empty();
  }

  private BGIParseResult<BGIData> parse() {
    if (originalData.length < 5) {
      return BGIParseResult.error(ErrorCode.DATA_TOO_SHORT);
    }

    if (originalData[0] != 7) {
      return BGIParseResult.error(ErrorCode.GOBLINLESS);
    }

    buffer.position(1); // Skip initial 7

    try {
      int dataVersion = ByteParser.getShort(buffer);
      var properties = parseProperties();
      return BGIParseResult.success(new BGIData(dataVersion, properties));
    } catch (BGIParseException e) {
      return BGIParseResult.error(e);
    }
  }

  private Map<Integer, Object> parseProperties() throws BGIParseException {
    int numAttributes = ByteParser.getShort(buffer);

    var properties = new HashMap<Integer, Object>();
    for (int i = 0; i < numAttributes; i++) {
      int key = ByteParser.getInt(buffer);
      try {
        Object value = ByteParser.getByType(key, buffer);
        properties.put(key, value);
      } catch (Exception e) {
        System.out.println(key + " " + e.getMessage());
        properties.put(key, null);
      }
    }

    return properties;
  }

  public static void main(String[] args) {
//    byte[] item1 = {
//        // Secrets Herbalism
////        7, 0, 1, 0, 11, -113, -89, -67, -85, 3, 4, 82, 65, 82, 69, -108, 65, 47, 58, 4, 0, -103, -106, 9, -23, 9, 25, 8, 69, 78, 67, 72, 65, 78, 84, 83, 1, 1, 8, 69, 78, 67, 72, 65, 78, 84, 83, 4, 0, 0, 0, 0, -98, -12, -13, -1, 2, 1, -88, -18, -67, -57, 3, 17, 83, 69, 67, 82, 69, 84, 83, 95, 72, 69, 82, 66, 65, 76, 73, 83, 77, 0, 35, -66, -10, 4, 6, 23, 84, 101, 97, 99, 104, 101, 115, 32, 121, 111, 117, 32, 115, 111, 109, 101, 32, 115, 101, 99, 114, 101, 116, 24, 104, 101, 114, 98, 97, 108, 105, 115, 109, 32, 107, 110, 111, 119, 108, 101, 100, 103, 101, 32, 119, 104, 101, 110, 9, 99, 111, 110, 115, 117, 109, 101, 100, 46, 0, 26, 38, 55, 73, 116, 39, 115, 32, 102, 117, 108, 108, 32, 111, 102, 32, 101, 120, 112, 101, 114, 105, 101, 110, 99, 101, 46, 21, 38, 55, 84, 97, 115, 116, 121, 32, 101, 120, 112, 101, 114, 105, 101, 110, 99, 101, 46, 46, 46, 0, 36, 114, -117, 3, 37, 60, 116, 105, 101, 114, 45, 99, 111, 108, 111, 114, 62, 84, 111, 109, 101, 32, 111, 102, 32, 72, 101, 114, 98, 97, 108, 105, 115, 109, 32, 83, 101, 99, 114, 101, 116, 115, 10, -69, -6, -120, 9, 59, 1, 1, 46, 109, 109, 111, 99, 111, 114, 101, 32, 97, 100, 109, 105, 110, 32, 101, 120, 112, 32, 103, 105, 118, 101, 32, 37, 112, 108, 97, 121, 101, 114, 37, 32, 104, 101, 114, 98, 97, 108, 105, 115, 109, 32, 53, 48, 48, 48, 0, 0, 0, 0, 0, 0, 0, 0, 1, 0, 12, 71, 3, -8, 3, 15, -62, -85, 32, 75, 110, 111, 119, 108, 101, 100, 103, 101, 32, -62, -69, 40, 59, -64, -26, 3, 10, 67, 79, 78, 83, 85, 77, 65, 66, 76, 69, 49, -114, -119, 96, 3, 10, 99, 111, 110, 115, 117, 109, 97, 98, 108, 101
//        // Runecarving Defense 5
////        7, 0, 1, 0, 23, -113, -89, -67, -85, 3, 12, 82, 85, 78, 69, 67, 65, 82, 86, 69, 95, 84, 53, -108, 65, 47, 58, 4, 0, -103, -106, 9, -23, 9, 25, 8, 69, 78, 67, 72, 65, 78, 84, 83, 1, 1, 8, 69, 78, 67, 72, 65, 78, 84, 83, 4, 0, 0, 0, 0, -98, -12, -13, -1, 2, 1, -96, -96, -120, 64, 9, 81, 24, 66, 76, 79, 67, 75, 95, 67, 79, 79, 76, 68, 79, 87, 78, 95, 82, 69, 68, 85, 67, 84, 73, 79, 78, 0, 0, 0, 1, 1, -95, -9, 22, -11, -70, -37, 69, 43, -120, -28, 54, 89, -83, -21, -4, 100, 1, 24, 66, 76, 79, 67, 75, 95, 67, 79, 79, 76, 68, 79, 87, 78, 95, 82, 69, 68, 85, 67, 84, 73, 79, 78, 8, 63, -31, -76, -105, 114, 74, 114, 6, -88, -18, -67, -57, 3, 21, 82, 85, 78, 69, 67, 65, 82, 86, 73, 78, 71, 95, 68, 69, 70, 69, 78, 83, 69, 95, 53, -83, -35, 78, 51, 3, 11, 82, 117, 110, 101, 99, 97, 114, 118, 105, 110, 103, -41, 121, -61, 20, 3, 5, 78, 69, 86, 69, 82, -17, -37, 114, -14, 9, 73, 20, 80, 86, 69, 95, 68, 65, 77, 65, 71, 69, 95, 82, 69, 68, 85, 67, 84, 73, 79, 78, 0, 0, 0, 1, 1, 11, 23, 108, 40, -106, 26, 72, -2, -109, 98, 36, -19, -2, -2, -40, -79, 1, 20, 80, 86, 69, 95, 68, 65, 77, 65, 71, 69, 95, 82, 69, 68, 85, 67, 84, 73, 79, 78, 8, 64, 6, -69, -53, 42, -14, 74, 98, -11, -6, 21, 89, 10, 1, 0, 35, -66, -10, 4, 5, 34, 38, 55, 68, 101, 102, 101, 110, 115, 105, 118, 101, 32, 109, 97, 103, 105, 99, 115, 32, 115, 119, 105, 114, 108, 32, 105, 110, 115, 105, 100, 101, 46, 46, 46, 0, 32, 38, 111, 72, 111, 108, 100, 32, 116, 104, 105, 115, 32, 111, 118, 101, 114, 32, 97, 110, 32, 105, 116, 101, 109, 32, 119, 105, 116, 104, 32, 97, 110, 38, 38, 111, 101, 109, 112, 116, 121, 32, 38, 97, 38, 111, 82, 117, 110, 101, 99, 97, 114, 118, 105, 110, 103, 32, 38, 53, 38, 111, 115, 111, 99, 107, 101, 116, 32, 97, 110, 100, 33, 38, 111, 114, 105, 103, 104, 116, 32, 99, 108, 105, 99, 107, 32, 116, 111, 32, 102, 105, 108, 108, 32, 116, 104, 101, 32, 115, 111, 99, 107, 101, 116, 33, 0, 36, 114, -117, 3, 29, 60, 116, 105, 101, 114, 45, 99, 111, 108, 111, 114, 62, 83, 101, 114, 101, 110, 105, 116, 121, 32, 45, 32, 82, 97, 110, 107, 32, 53, 0, 39, 73, -30, 8, 64, 20, 0, 0, 0, 0, 0, 0, 12, 71, 3, -8, 3, 22, -62, -85, 32, 82, 117, 110, 101, 99, 97, 114, 118, 105, 110, 103, 32, 82, 117, 110, 101, 32, -62, -69, 16, -73, 70, -64, 10, 1, 40, 59, -64, -26, 3, 11, 82, 85, 78, 69, 67, 65, 82, 86, 73, 78, 71, 55, -114, -17, 85, 9, 65, 16, 72, 69, 65, 76, 73, 78, 71, 95, 82, 69, 67, 69, 73, 86, 69, 68, 0, 0, 0, 1, 1, 15, -75, -118, 58, 23, 6, 75, -124, -91, -46, -23, -13, 52, 15, -61, 22, 1, 16, 72, 69, 65, 76, 73, 78, 71, 95, 82, 69, 67, 69, 73, 86, 69, 68, 8, 64, 19, 80, -114, -5, 99, -81, -82, 69, 79, 92, -34, 9, 57, 12, 66, 76, 79, 67, 75, 95, 82, 65, 84, 73, 78, 71, 0, 0, 0, 1, 1, -120, 111, -83, 109, -68, 60, 72, 19, -125, -124, -6, 105, -105, -13, 82, -75, 1, 12, 66, 76, 79, 67, 75, 95, 82, 65, 84, 73, 78, 71, 8, 63, -2, -101, 79, -114, 66, -37, -9, 77, 62, -83, -111, 9, 47, 7, 68, 69, 70, 69, 78, 83, 69, 0, 0, 0, 1, 1, 127, -109, 99, 101, 32, -107, 65, -91, -88, -111, 91, 51, -36, 33, 58, 77, 1, 7, 68, 69, 70, 69, 78, 83, 69, 8, 64, 9, 43, -60, 54, 50, -84, 68, 86, 20, -58, 71, 10, 1, 109, -127, 109, -124, 9, 55, 11, 66, 76, 79, 67, 75, 95, 80, 79, 87, 69, 82, 0, 0, 0, 1, 1, 6, 48, -32, -82, -19, 67, 78, -11, -128, -74, -11, 50, -12, 8, -76, 103, 1, 11, 66, 76, 79, 67, 75, 95, 80, 79, 87, 69, 82, 8, 63, -1, 114, 15, 87, -81, -115, -82, 114, -75, -86, -122, 10, 1, 121, -12, 84, -106, 10, 1
//        // Runecarving Empowered
////        7, 0, 1, 0, 20, -113, -89, -67, -85, 3, 12, 82, 85, 78, 69, 67, 65, 82, 86, 69, 95, 84, 51, -108, 65, 47, 58, 4, 0, -103, -106, 9, -23, 9, 25, 8, 69, 78, 67, 72, 65, 78, 84, 83, 1, 1, 8, 69, 78, 67, 72, 65, 78, 84, 83, 4, 0, 0, 0, 0, -98, -12, -13, -1, 2, 1, -88, -18, -67, -57, 3, 23, 82, 85, 78, 69, 67, 65, 82, 86, 73, 78, 71, 95, 69, 77, 80, 79, 87, 69, 82, 69, 68, 95, 51, -87, -102, -44, 123, 9, 100, 7, 65, 66, 73, 76, 73, 84, 89, 0, 0, 0, 1, 1, -38, -25, -50, -86, -77, -50, 74, -101, -115, -9, 104, 25, 52, -71, -45, -59, 1, 7, 65, 66, 73, 76, 73, 84, 89, 9, 60, 1, 1, 9, 83, 87, 73, 70, 84, 78, 69, 83, 83, 11, 75, 73, 76, 76, 95, 69, 78, 84, 73, 84, 89, 2, 8, 100, 117, 114, 97, 116, 105, 111, 110, 64, 17, -48, 22, -31, -18, -101, -36, 9, 97, 109, 112, 108, 105, 102, 105, 101, 114, 64, 8, 0, 0, 0, 0, 0, 0, -83, -35, 78, 51, 3, 9, 69, 109, 112, 111, 119, 101, 114, 101, 100, -41, 121, -61, 20, 3, 5, 78, 69, 86, 69, 82, -29, 61, 65, 10, 9, 60, 1, 1, 9, 83, 87, 73, 70, 84, 78, 69, 83, 83, 11, 75, 73, 76, 76, 95, 69, 78, 84, 73, 84, 89, 2, 8, 100, 117, 114, 97, 116, 105, 111, 110, 64, 17, -48, 22, -31, -18, -101, -36, 9, 97, 109, 112, 108, 105, 102, 105, 101, 114, 64, 8, 0, 0, 0, 0, 0, 0, -11, -6, 21, 89, 10, 1, 0, 35, -66, -10, 4, 5, 33, 38, 55, 80, 111, 119, 101, 114, 102, 117, 108, 32, 109, 97, 103, 105, 99, 115, 32, 115, 119, 105, 114, 108, 32, 105, 110, 115, 105, 100, 101, 46, 46, 46, 0, 32, 38, 111, 72, 111, 108, 100, 32, 116, 104, 105, 115, 32, 111, 118, 101, 114, 32, 97, 110, 32, 105, 116, 101, 109, 32, 119, 105, 116, 104, 32, 97, 110, 36, 38, 111, 101, 109, 112, 116, 121, 32, 38, 97, 38, 111, 69, 109, 112, 111, 119, 101, 114, 101, 100, 32, 38, 53, 38, 111, 115, 111, 99, 107, 101, 116, 32, 97, 110, 100, 33, 38, 111, 114, 105, 103, 104, 116, 32, 99, 108, 105, 99, 107, 32, 116, 111, 32, 102, 105, 108, 108, 32, 116, 104, 101, 32, 115, 111, 99, 107, 101, 116, 33, 0, 36, 114, -117, 3, 31, 60, 116, 105, 101, 114, 45, 99, 111, 108, 111, 114, 62, 65, 100, 114, 101, 110, 97, 108, 105, 110, 101, 32, 45, 32, 82, 97, 110, 107, 32, 51, 0, 39, 73, -30, 8, 64, 8, 0, 0, 0, 0, 0, 0, 12, 71, 3, -8, 3, 20, -62, -85, 32, 69, 109, 112, 111, 119, 101, 114, 101, 100, 32, 82, 117, 110, 101, 32, -62, -69, 16, -73, 70, -64, 10, 1, 40, 59, -64, -26, 3, 11, 82, 85, 78, 69, 67, 65, 82, 86, 73, 78, 71, 86, 20, -58, 71, 10, 1, 107, -73, -4, -70, 9, 100, 4, 78, 65, 77, 69, 1, 1, 4, 78, 65, 77, 69, 3, 27, 60, 116, 105, 101, 114, 45, 99, 111, 108, 111, 114, 62, 80, 111, 119, 101, 114, 115, 32, 45, 32, 82, 97, 110, 107, 32, 51, 0, 0, 1, 1, -38, -25, -50, -86, -77, -50, 74, -101, -115, -9, 104, 25, 52, -71, -45, -59, 1, 4, 78, 65, 77, 69, 3, 31, 60, 116, 105, 101, 114, 45, 99, 111, 108, 111, 114, 62, 65, 100, 114, 101, 110, 97, 108, 105, 110, 101, 32, 45, 32, 82, 97, 110, 107, 32, 51, 114, -75, -86, -122, 10, 1, 121, -12, 84, -106, 10, 1
//        // Skeletal Bow
////        7, 0, 1, 0, 23, -113, -89, -67, -85, 3, 13, 85, 78, 67, 79, 77, 77, 79, 78, 95, 71, 69, 65, 82, -108, 65, 47, 58, 4, 1, 10, 105, 110, 102, 105, 110, 105, 116, 121, 32, 49, -103, -106, 9, -23, 9, 36, 8, 69, 78, 67, 72, 65, 78, 84, 83, 1, 1, 8, 69, 78, 67, 72, 65, 78, 84, 83, 4, 1, 10, 105, 110, 102, 105, 110, 105, 116, 121, 32, 49, 0, 0, 0, -98, -12, -13, -1, 2, 2, -88, -18, -67, -57, 3, 12, 83, 75, 69, 76, 69, 84, 65, 76, 95, 66, 79, 87, -84, -126, 1, -28, 2, -24, 7, -71, -35, 14, -63, 10, 1, -27, -110, 92, -84, 3, 14, 80, 79, 87, 68, 69, 82, 124, 67, 79, 82, 69, 95, 84, 50, -18, -43, -60, -57, 8, 64, 36, 0, 0, 0, 0, 0, 0, -11, -6, 21, 89, 10, 1, -7, -91, 118, 48, 9, 15, 1, 1, 11, 82, 117, 110, 101, 99, 97, 114, 118, 105, 110, 103, 0, 0, 35, -66, -10, 4, 2, 18, 38, 55, 83, 116, 114, 105, 110, 103, 32, 111, 102, 32, 115, 105, 110, 101, 119, 46, 15, 38, 55, 71, 114, 105, 112, 32, 111, 102, 32, 98, 111, 110, 101, 46, 0, 36, 114, -117, 3, 24, 60, 116, 105, 101, 114, 45, 99, 111, 108, 111, 114, 62, 83, 107, 101, 108, 101, 116, 97, 108, 32, 66, 111, 119, 0, 39, 73, -30, 8, 64, 0, 0, 0, 0, 0, 0, 0, 1, 92, -96, 29, 8, 64, 20, 0, 0, 0, 0, 0, 0, 16, -73, 70, -64, 10, 1, 40, 59, -64, -26, 3, 3, 66, 79, 87, 70, 121, -42, -72, 8, 64, 53, -36, -25, -37, 13, -53, -82, 77, 81, -7, 70, 8, 64, 44, -5, 78, -118, 15, 109, -42, 107, 77, -39, -73, 9, 26, 1, 6, 83, 78, 69, 69, 90, 69, 5, 63, -87, -103, -103, -103, -103, -103, -102, 0, 63, -87, -103, -103, -103, -103, -103, -102, 114, 37, 123, -63, 8, 64, 23, 71, -94, -1, 123, 93, 64, 114, -75, -86, -122, 10, 1, 121, -12, 84, -106, 10, 1
//        // Elixir Mega Pint (Herbalism)
////        7, 0, 1, 0, 11, -113, -89, -67, -85, 3, 6, 69, 88, 79, 84, 73, 67, -108, 65, 47, 58, 4, 0, -103, -106, 9, -23, 9, 25, 8, 69, 78, 67, 72, 65, 78, 84, 83, 1, 1, 8, 69, 78, 67, 72, 65, 78, 84, 83, 4, 0, 0, 0, 0, -98, -12, -13, -1, 2, 1, -88, -18, -67, -57, 3, 25, 69, 76, 73, 88, 73, 82, 95, 72, 69, 82, 66, 65, 76, 73, 83, 77, 95, 77, 69, 71, 65, 80, 73, 78, 84, 0, 35, -66, -10, 4, 4, 34, 38, 55, 89, 111, 117, 32, 112, 111, 117, 114, 101, 100, 32, 121, 111, 117, 114, 115, 101, 108, 102, 32, 97, 32, 77, 101, 103, 97, 45, 80, 105, 110, 116, 63, 54, 38, 55, 73, 32, 112, 111, 117, 114, 101, 100, 32, 109, 121, 115, 101, 108, 102, 32, 97, 32, 108, 97, 114, 103, 101, 32, 70, 108, 97, 115, 107, 32, 111, 102, 32, 66, 111, 117, 110, 116, 105, 102, 117, 108, 32, 72, 97, 114, 118, 101, 115, 116, 115, 46, 0, 40, 38, 99, 82, 101, 112, 108, 97, 99, 101, 115, 32, 97, 110, 121, 32, 97, 99, 116, 105, 118, 101, 32, -62, -85, 32, 69, 108, 105, 120, 105, 114, 32, -62, -69, 32, 98, 117, 102, 102, 46, 0, 36, 114, -117, 3, 43, 60, 116, 105, 101, 114, 45, 99, 111, 108, 111, 114, 62, 77, 101, 103, 97, 45, 80, 105, 110, 116, 32, 111, 102, 32, 66, 111, 117, 110, 116, 105, 102, 117, 108, 32, 72, 97, 114, 118, 101, 115, 116, 115, 12, 71, 3, -8, 3, 12, -62, -85, 32, 69, 108, 105, 120, 105, 114, 32, -62, -69, 40, 59, -64, -26, 3, 10, 67, 79, 78, 83, 85, 77, 65, 66, 76, 69, 49, -114, -119, 96, 3, 10, 99, 111, 110, 115, 117, 109, 97, 98, 108, 101, 63, -50, -39, -124, 9, -98, 1, 1, 11, 66, 85, 70, 70, 95, 69, 76, 73, 88, 73, 82, 2, 63, -12, 0, 0, 0, 0, 0, 0, 64, -84, 32, 0, 0, 0, 0, 0, 31, 65, 68, 68, 73, 84, 73, 79, 78, 65, 76, 95, 69, 88, 80, 69, 82, 73, 69, 78, 67, 69, 95, 72, 69, 82, 66, 65, 76, 73, 83, 77, 11, 66, 85, 70, 70, 95, 69, 76, 73, 88, 73, 82, 14, 109, 117, 108, 116, 105, 112, 108, 105, 99, 97, 116, 105, 118, 101, 63, -16, 0, 0, 0, 0, 0, 0, 64, 20, 0, 0, 0, 0, 0, 0, 64, -84, 32, 0, 0, 0, 0, 0, 15, 83, 75, 73, 76, 76, 95, 72, 69, 82, 66, 65, 76, 73, 83, 77, 11, 66, 85, 70, 70, 95, 69, 76, 73, 88, 73, 82, 8, 97, 100, 100, 105, 116, 105, 118, 101, 63, -16, 0, 0, 0, 0, 0, 0
//        // Alchemist Chestplate
////        7, 0, 1, 0, 22, -122, -31, 26, 32, 8, 64, 64, 37, 55, 78, -5, -98, 82, -113, -89, -67, -85, 3, 9, 82, 65, 82, 69, 95, 71, 69, 65, 82, -108, 65, 47, 58, 4, 0, -103, -106, 9, -23, 9, 25, 8, 69, 78, 67, 72, 65, 78, 84, 83, 1, 1, 8, 69, 78, 67, 72, 65, 78, 84, 83, 4, 0, 0, 0, 0, -100, 110, -90, 112, 4, 1, 22, 38, 54, 111, 102, 32, 116, 104, 101, 32, 80, 111, 116, 105, 111, 110, 32, 83, 101, 108, 108, 101, 114, -98, -12, -13, -1, 2, 1, -88, -18, -67, -57, 3, 22, 65, 76, 67, 72, 69, 77, 73, 83, 84, 95, 67, 72, 69, 83, 84, 80, 76, 65, 84, 69, 95, 52, -84, -126, 1, -28, 2, -24, 7, -78, -1, -102, 102, 8, 64, 62, 0, 0, 0, 0, 0, 0, -27, -110, 92, -84, 3, 14, 80, 79, 87, 68, 69, 82, 124, 67, 79, 82, 69, 95, 84, 52, -17, 17, -96, -92, 3, 4, 105, 114, 111, 110, -7, -91, 118, 48, 9, 27, 1, 2, 11, 82, 117, 110, 101, 99, 97, 114, 118, 105, 110, 103, 11, 82, 117, 110, 101, 99, 97, 114, 118, 105, 110, 103, 0, 0, 1, 64, 66, 3, 13, 83, 69, 84, 95, 65, 76, 67, 72, 69, 77, 73, 83, 84, 0, 35, -66, -10, 4, 1, 45, 38, 55, 70, 97, 110, 99, 121, 32, 97, 110, 100, 32, 100, 101, 102, 105, 110, 105, 116, 101, 108, 121, 32, 110, 111, 116, 32, 99, 111, 118, 101, 114, 101, 100, 32, 105, 110, 32, 115, 116, 97, 105, 110, 115, 46, 0, 36, 114, -117, 3, 39, 60, 116, 105, 101, 114, 45, 99, 111, 108, 111, 114, 62, 69, 120, 112, 101, 114, 116, 32, 65, 108, 99, 104, 101, 109, 105, 115, 116, 115, 32, 86, 101, 115, 116, 109, 101, 110, 116, 115, 0, 39, 73, -30, 8, 64, 16, 0, 0, 0, 0, 0, 0, 1, 70, 72, 83, 3, 3, 118, 101, 120, 12, 71, 3, -8, 3, 19, -62, -85, 32, 65, 108, 99, 104, 101, 109, 121, 32, 65, 114, 109, 111, 114, 32, -62, -69, 40, 59, -64, -26, 3, 5, 65, 82, 77, 79, 82, 59, 92, -74, 36, 8, 64, 86, -117, -125, 117, 34, -108, 46, 104, -34, -82, -43, 9, -125, 1, 29, 65, 68, 68, 73, 84, 73, 79, 78, 65, 76, 95, 69, 88, 80, 69, 82, 73, 69, 78, 67, 69, 95, 65, 76, 67, 72, 69, 77, 89, 1, 1, 29, 65, 68, 68, 73, 84, 73, 79, 78, 65, 76, 95, 69, 88, 80, 69, 82, 73, 69, 78, 67, 69, 95, 65, 76, 67, 72, 69, 77, 89, 8, 64, 84, 7, 119, 34, -37, -16, -109, 0, 0, 1, 1, 27, -97, 98, -54, -29, 75, 78, -11, -99, 127, 88, 127, 12, -106, -72, 53, 1, 29, 65, 68, 68, 73, 84, 73, 79, 78, 65, 76, 95, 69, 88, 80, 69, 82, 73, 69, 78, 67, 69, 95, 65, 76, 67, 72, 69, 77, 89, 8, 64, 36, 32, 98, -110, 53, 28, -36, 107, -73, -4, -70, 9, 115, 4, 78, 65, 77, 69, 1, 1, 4, 78, 65, 77, 69, 3, 39, 60, 116, 105, 101, 114, 45, 99, 111, 108, 111, 114, 62, 69, 120, 112, 101, 114, 116, 32, 65, 108, 99, 104, 101, 109, 105, 115, 116, 115, 32, 86, 101, 115, 116, 109, 101, 110, 116, 115, 0, 0, 1, 1, 27, -97, 98, -54, -29, 75, 78, -11, -99, 127, 88, 127, 12, -106, -72, 53, 2, 8, 78, 65, 77, 69, 95, 83, 85, 70, 4, 1, 22, 38, 54, 111, 102, 32, 116, 104, 101, 32, 80, 111, 116, 105, 111, 110, 32, 83, 101, 108, 108, 101, 114, 4, 78, 65, 77, 69, 3, 0
//        // Brass Chisel
//        7, 0, 1, 0, 27, -122, -31, 26, 32, 8, 64, 24, 41, 15, -7, 114, 71, 69, -113, -89, -67, -85, 3, 13, 85, 78, 67, 79, 77, 77, 79, 78, 95, 71, 69, 65, 82, -108, 65, 47, 58, 4, 0, -103, -106, 9, -23, 9, 25, 8, 69, 78, 67, 72, 65, 78, 84, 83, 1, 1, 8, 69, 78, 67, 72, 65, 78, 84, 83, 4, 0, 0, 0, 0, -100, 110, -90, 112, 4, 1, 21, 38, 54, 111, 102, 32, 116, 104, 101, 32, 69, 97, 114, 116, 104, 98, 114, 101, 97, 107, 101, 114, -98, -12, -13, -1, 2, 1, -90, 16, 27, -116, 9, 73, 20, 75, 78, 79, 67, 75, 66, 65, 67, 75, 95, 82, 69, 83, 73, 83, 84, 65, 78, 67, 69, 0, 1, 1, 7, -20, -122, 61, -71, -26, 65, 44, -65, -103, -41, -123, -92, -110, 20, -75, 1, 20, 75, 78, 79, 67, 75, 66, 65, 67, 75, 95, 82, 69, 83, 73, 83, 84, 65, 78, 67, 69, 8, 63, -124, 122, -31, 71, -82, 20, 123, 0, 0, -88, -18, -67, -57, 3, 16, 65, 85, 71, 77, 69, 78, 84, 95, 77, 73, 78, 73, 78, 71, 95, 50, -84, -126, 1, -28, 2, -24, 7, -54, -78, -18, 30, 8, 64, 16, 0, 0, 0, 0, 0, 0, -35, -24, 119, -36, 9, -128, 1, 28, 65, 68, 68, 73, 84, 73, 79, 78, 65, 76, 95, 69, 88, 80, 69, 82, 73, 69, 78, 67, 69, 95, 77, 73, 78, 73, 78, 71, 1, 1, 28, 65, 68, 68, 73, 84, 73, 79, 78, 65, 76, 95, 69, 88, 80, 69, 82, 73, 69, 78, 67, 69, 95, 77, 73, 78, 73, 78, 71, 8, 64, 52, 0, 0, 0, 0, 0, 0, 0, 0, 1, 1, -2, 80, -37, -102, 105, -127, 76, -55, -120, 58, -91, 58, -100, -118, -97, 15, 1, 28, 65, 68, 68, 73, 84, 73, 79, 78, 65, 76, 95, 69, 88, 80, 69, 82, 73, 69, 78, 67, 69, 95, 77, 73, 78, 73, 78, 71, 8, 64, 18, -69, 47, -20, 86, -43, -48, -27, -110, 92, -84, 3, 14, 80, 79, 87, 68, 69, 82, 124, 67, 79, 82, 69, 95, 84, 50, -20, -12, -54, -83, 8, 64, 56, -82, -53, -5, 21, -75, 116, -7, -91, 118, 48, 9, 88, 1, 0, 1, 19, -62, -89, 102, 77, 117, 100, 100, 108, 101, 100, 32, 45, 32, 82, 97, 110, 107, 32, 49, 7, -20, -122, 61, -71, -26, 65, 44, -65, -103, -41, -123, -92, -110, 20, -75, 1, 21, 82, 85, 78, 69, 67, 65, 82, 86, 73, 78, 71, 95, 77, 85, 68, 68, 76, 69, 68, 95, 49, 1, 11, 82, 85, 78, 69, 67, 65, 82, 86, 73, 78, 71, 0, 11, 82, 117, 110, 101, 99, 97, 114, 118, 105, 110, 103, 0, 1, 64, 66, 3, 9, 83, 69, 84, 95, 77, 73, 78, 69, 82, 0, 35, -66, -10, 4, 1, 30, 38, 55, 87, 65, 82, 78, 73, 78, 71, 58, 32, 65, 118, 111, 105, 100, 32, 84, 104, 117, 110, 100, 101, 114, 115, 116, 111, 114, 109, 115, 0, 36, 114, -117, 3, 24, 60, 116, 105, 101, 114, 45, 99, 111, 108, 111, 114, 62, 66, 114, 97, 115, 115, 32, 67, 104, 105, 115, 101, 108, 0, 39, 73, -30, 8, 64, 0, 0, 0, 0, 0, 0, 0, 12, 71, 3, -8, 3, 20, -62, -85, 32, 77, 105, 110, 105, 110, 103, 32, 65, 117, 103, 109, 101, 110, 116, 32, -62, -69, 40, 59, -64, -26, 3, 7, 65, 85, 71, 77, 69, 78, 84, 46, -54, -88, 3, 10, 1, 59, 35, 86, 43, 8, 64, 36, 0, 0, 0, 0, 0, 0, 63, -98, -37, -83, 8, 63, -6, -112, -1, -105, 36, 116, 84, 95, -96, -49, 93, 8, 63, -124, 122, -31, 71, -82, 20, 123, 102, -36, -126, -5, 10, 1, 107, -73, -4, -70, 9, 99, 4, 78, 65, 77, 69, 1, 1, 4, 78, 65, 77, 69, 3, 24, 60, 116, 105, 101, 114, 45, 99, 111, 108, 111, 114, 62, 66, 114, 97, 115, 115, 32, 67, 104, 105, 115, 101, 108, 0, 0, 1, 1, -2, 80, -37, -102, 105, -127, 76, -55, -120, 58, -91, 58, -100, -118, -97, 15, 2, 8, 78, 65, 77, 69, 95, 83, 85, 70, 4, 1, 21, 38, 54, 111, 102, 32, 116, 104, 101, 32, 69, 97, 114, 116, 104, 98, 114, 101, 97, 107, 101, 114, 4, 78, 65, 77, 69, 3, 0, 109, 32, -44, 94, 9, 91, 29, 65, 68, 68, 73, 84, 73, 79, 78, 65, 76, 95, 69, 88, 80, 69, 82, 73, 69, 78, 67, 69, 95, 70, 73, 83, 72, 73, 78, 71, 0, 1, 1, 7, -20, -122, 61, -71, -26, 65, 44, -65, -103, -41, -123, -92, -110, 20, -75, 1, 29, 65, 68, 68, 73, 84, 73, 79, 78, 65, 76, 95, 69, 88, 80, 69, 82, 73, 69, 78, 67, 69, 95, 70, 73, 83, 72, 73, 78, 71, 8, 63, -6, -112, -1, -105, 36, 116, 84, 0, 0
//    };
//    BGIParseResult<BGIData> result = BGIParser.parse(item1);
//    if (result.isSuccess()) {
//      var data = result.result().get();
//
//      System.out.println("Num Attributes: " + data.getNumAttributes());
//
//      var attrs = data.getAttributes();
//      for (var key : attrs.keySet()) {
//        boolean found = false;
//        for (BGIField field : BGIField.values()) {
//          if (field.name().hashCode() == key) {
//            var value = attrs.get(key);
//
//            if (value instanceof String[]) {
//              System.out.println(field + " = " + Arrays.toString((String[])value));
//            } else {
//              System.out.println(field + " = " + value);
//            }
//            found = true;
//            break;
//          }
//        }
//
//        if (!found) {
//          System.out.println(key + " = " + attrs.get(key));
//        }
//      }
//    } else {
//      System.out.println(result.error().get().getMessage());
//    }
  }
}
