package dev.bnjc.bglib;

import dev.bnjc.bglib.exceptions.BGIParseException;
import dev.bnjc.bglib.exceptions.ErrorCode;
import java.nio.ByteBuffer;
import java.nio.ByteOrder;
import java.nio.charset.StandardCharsets;
import java.util.HashMap;
import java.util.Map;
import net.minecraft.class_1799;
import net.minecraft.class_2487;
import net.minecraft.class_2520;
import net.minecraft.class_9279;
import net.minecraft.class_9334;

/**
 * A parser to parse the custom Blockgame Item (BGI) tag found in the custom data component.
 *
 * <pre>
 *   BGIResult result = BGIParser.parse(itemStack);
 *   result.ifSuccess((data) -> {
 *     String itemId = data.getString(BGIField.ITEM_ID);
 *   });
 * </pre>
 *
 * @author Jack Grzechowiak
 * @since 0.1.2
 */
public final class BGIParser {
  /**
   * The name of the Blockgame Item byte array tag used within an {@linkplain class_1799} custom data component
   */
  public static final String BGI_TAG = "bgi";

  private final ByteBuffer buffer;
  private final byte[] originalData;

  private BGIParser(byte[] data) {
    this.originalData = data;
    this.buffer = ByteBuffer.wrap(data);
    this.buffer.order(ByteOrder.BIG_ENDIAN);
  }

  /**
   * Parses the specified byte array into a {@link BGIParseResult} object.
   *
   * @param data Byte array from NBT data
   * @return a {@link BGIParseResult} corresponding to the specified byte array
   * @since 0.1.1
   */
  public static BGIParseResult<BGIData> parse(byte[] data) {
    var parser = new BGIParser(data);
    return parser.parse();
  }

  /**
   * Parses the specified item stack's custom data component into a {@link BGIParseResult} object. If the stack
   * does not have the "bgi" tag in its custom data, then an error will be returned.
   *
   * @param itemStack Item stack to find the tag in
   * @return a {@link BGIParseResult} corresponding to the specified item stack
   */
  public static BGIParseResult<BGIData> parse(class_1799 itemStack) {
    class_2487 stackNbt = itemStack.method_57825(class_9334.field_49628, class_9279.field_49302).method_57461();

    if (stackNbt != null && stackNbt.method_10573(BGI_TAG, class_2520.field_33257)) {
      return BGIParser.parse(stackNbt.method_10547(BGI_TAG));
    }

    return BGIParseResult.error(ErrorCode.MISSING_TAG);
  }

  private BGIParseResult<BGIData> parse() {
    if (originalData.length < 5) {
      return BGIParseResult.error(ErrorCode.DATA_TOO_SHORT);
    }

    if (originalData[0] != 7) {
      return BGIParseResult.error(ErrorCode.GOBLINLESS);
    }

    buffer.position(1); // Skip initial 7

    try {
      int dataVersion = getShort();
      var properties = parseProperties();
      return BGIParseResult.success(new BGIData(dataVersion, properties));
    } catch (BGIParseException e) {
      return BGIParseResult.error(e);
    }
  }

  private Map<Integer, Object> parseProperties() throws BGIParseException {
    int numAttributes = getShort();

    var properties = new HashMap<Integer, Object>();
    for (int i = 0; i < numAttributes; i++) {
      int key = getInt();
      byte typeId = getByte();

      Object value = switch (typeId) {
        case 2 -> getVarInt();
        case 3 -> getString();
        case 4 -> getStringArray();
        case 8 -> getDouble();
        case 9 -> getStream();
        case 10 -> getBoolean();
        default -> throw new BGIParseException("Could not parse data type [" + typeId + "]", ErrorCode.UNKNOWN_DATA_TYPE);
      };

      properties.put(key, value);
    }

    return properties;
  }

  private byte getByte() {
    return buffer.get();
  }

  private short getShort() {
    return buffer.getShort();
  }

  private double getDouble() {
    return buffer.getDouble();
  }

  private boolean getBoolean() {
    return getByte() != 0;
  }

  private String getString() {
    int length = getVarInt();
    byte[] strBytes = new byte[length];
    buffer.get(strBytes);
    return new String(strBytes, StandardCharsets.UTF_8);
  }

  private String[] getStringArray() {
    int count = getVarInt();
    String[] array = new String[count];
    for (int j = 0; j < count; j++) {
      array[j] = getString();
    }
    return array;
  }

  private byte[] getStream() {
    int length = getVarInt();
    byte[] streamBytes = new byte[length];
    buffer.get(streamBytes);
    return streamBytes;
  }

  private int getInt() {
    return buffer.getInt();
  }

  private int getVarInt() {
    int value = 0;
    int shift = 0;

    while (buffer.hasRemaining()) {
      byte b = getByte();

      // Get the 7 data bits
      int temp = b & 0x7F;
      value |= temp << shift;

      // If the continuation bit is not set, we're done
      if ((b & 0x80) == 0) {
        break;
      }

      // Each byte contributes 7 bits
      shift += 7;

      // guard against malformed input
      if (shift > 32) {
        throw new IllegalArgumentException("Malformed variable int");
      }
    }

    return value;
  }

  public static void main(String[] args) {
    byte[] item1 = {7, 0, 1, 0, 13, -113, -89, -67, -85, 3, 6, 67, 79, 77, 77, 79, 78, -108, 65, 47, 58, 4, 0, -103, -106, 9, -23, 9, 25, 8, 69, 78, 67, 72, 65, 78, 84, 83, 1, 1,
        8, 69, 78, 67, 72, 65, 78, 84, 83, 4, 0, 0, 0, 0, -98, -12, -13, -1, 2, 1, -88, -18, -67, -57, 3, 20, 87, 69, 76, 76, 95, 83, 69, 65, 83, 79, 78, 69, 68, 95,
        66, 69, 69, 70, 95, 49, -48, 7, 51, 73, 3, 20, 87, 69, 76, 76, 95, 83, 69, 65, 83, 79, 78, 69, 68, 95, 66, 69, 69, 70, 95, 49, 0, 35, -66, -10, 4, 3, 17, 38,
        55, 77, 97, 107, 101, 115, 32, 121, 97, 32, 66, 69, 69, 70, 89, 46, 0, 38, 38, 99, 82, 101, 112, 108, 97, 99, 101, 115, 32, 97, 110, 121, 32, 97, 99, 116, 105, 118, 101,
        32, -62, -85, 32, 70, 111, 111, 100, 32, -62, -69, 32, 98, 117, 102, 102, 46, 0, 36, 114, -117, 3, 32, 60, 116, 105, 101, 114, 45, 99, 111, 108, 111, 114, 62, 87, 101, 108, 108,
        45, 83, 101, 97, 115, 111, 110, 101, 100, 32, 66, 101, 101, 102, 32, 73, 9, -81, 127, 78, 8, 64, 16, 0, 0, 0, 0, 0, 0, 12, 71, 3, -8, 3, 10, -62, -85, 32, 70,
        111, 111, 100, 32, -62, -69, 40, 59, -64, -26, 3, 10, 67, 79, 78, 83, 85, 77, 65, 66, 76, 69, 49, -114, -119, 96, 3, 10, 99, 111, 110, 115, 117, 109, 97, 98, 108, 101, 63, -50,
        -39, -124, 9, -69, 1, 1, 9, 66, 85, 70, 70, 95, 70, 79, 79, 68, 3, -64, 0, 0, 0, 0, 0, 0, 0, 64, -100, 32, 0, 0, 0, 0, 0, 14, 82, 69, 68, 85, 67, 69,
        68, 95, 72, 85, 78, 71, 69, 82, 9, 66, 85, 70, 70, 95, 70, 79, 79, 68, 8, 97, 100, 100, 105, 116, 105, 118, 101, 63, -16, 0, 0, 0, 0, 0, 0, 64, 36, 0, 0,
        0, 0, 0, 0, 64, -100, 32, 0, 0, 0, 0, 0, 10, 77, 65, 88, 95, 72, 69, 65, 76, 84, 72, 9, 66, 85, 70, 70, 95, 70, 79, 79, 68, 8, 97, 100, 100, 105, 116,
        105, 118, 101, 63, -16, 0, 0, 0, 0, 0, 0, 63, -71, -103, -103, -103, -103, -103, -102, 64, -100, 32, 0, 0, 0, 0, 0, 19, 72, 69, 65, 76, 84, 72, 95, 82, 69, 71, 69,
        78, 69, 82, 65, 84, 73, 79, 78, 9, 66, 85, 70, 70, 95, 70, 79, 79, 68, 8, 97, 100, 100, 105, 116, 105, 118, 101, 63, -16, 0, 0, 0, 0, 0, 0};
    BGIParseResult<BGIData> result = BGIParser.parse(item1);
    if (result.isSuccess()) {
      var data = result.result().get();
      System.out.println("Num Attributes: " + data.getNumAttributes());

      var attrs = data.getAttributes();
      System.out.println(attrs.get(-804834487));
      for (var key : attrs.keySet()) {
        boolean found = false;
        for (BGIField field : BGIField.values()) {
          if (field.name().hashCode() == key) {
            System.out.println(field + " = " + attrs.get(key));
            found = true;
            break;
          }
        }

        if (!found) {
          System.out.println(key + " = " + attrs.get(key));
        }
      }
    }
  }

//  public static byte[] hexStringToByteArray(String hex) {
//    int len = hex.length();
//    if (len % 2 != 0) {
//      throw new IllegalArgumentException("Hex string must have even length");
//    }
//    byte[] result = new byte[len / 2];
//    for (int i = 0; i < len; i += 2) {
//      result[i / 2] = (byte) ((Character.digit(hex.charAt(i), 16) << 4)
//          + Character.digit(hex.charAt(i+1), 16));
//    }
//    return result;
//  }
}
