package dev.bnjc.bglib;

import dev.bnjc.bglib.exceptions.BGIParseException;
import dev.bnjc.bglib.exceptions.ErrorCode;
import java.nio.ByteBuffer;
import java.nio.ByteOrder;
import java.nio.charset.StandardCharsets;
import java.util.HashMap;
import java.util.Map;
import net.minecraft.class_1799;
import net.minecraft.class_2487;
import net.minecraft.class_2520;
import net.minecraft.class_9279;
import net.minecraft.class_9334;

/**
 * A parser to parse the custom Blockgame Item (BGI) tag found in the custom data component.
 *
 * <pre>
 *   BGIResult result = BGIParser.parse(itemStack);
 *   result.ifSuccess((data) -> {
 *     String itemId = data.getString(BGIField.ITEM_ID);
 *   });
 * </pre>
 *
 * @author Jack Grzechowiak
 * @since 0.1.2
 */
public final class BGIParser {
  /**
   * The name of the Blockgame Item byte array tag used within an {@linkplain class_1799} custom data component
   */
  public static final String BGI_TAG = "bgi";

  private final ByteBuffer buffer;
  private final byte[] originalData;

  private BGIParser(byte[] data) {
    this.originalData = data;
    this.buffer = ByteBuffer.wrap(data);
    this.buffer.order(ByteOrder.BIG_ENDIAN);
  }

  /**
   * Parses the specified byte array into a {@link BGIParseResult} object.
   *
   * @param data Byte array from NBT data
   * @return a {@link BGIParseResult} corresponding to the specified byte array
   * @since 0.1.1
   */
  public static BGIParseResult<BGIData> parse(byte[] data) {
    var parser = new BGIParser(data);
    return parser.parse();
  }

  /**
   * Parses the specified item stack's custom data component into a {@link BGIParseResult} object. If the stack
   * does not have the "bgi" tag in its custom data, then an error will be returned.
   *
   * @param itemStack Item stack to find the tag in
   * @return a {@link BGIParseResult} corresponding to the specified item stack
   */
  public static BGIParseResult<BGIData> parse(class_1799 itemStack) {
    class_2487 stackNbt = itemStack.method_57825(class_9334.field_49628, class_9279.field_49302).method_57461();

    if (stackNbt != null && stackNbt.method_10573(BGI_TAG, class_2520.field_33257)) {
      return BGIParser.parse(stackNbt.method_10547(BGI_TAG));
    }

    return BGIParseResult.error(ErrorCode.MISSING_TAG);
  }

  private BGIParseResult<BGIData> parse() {
    if (originalData.length < 5) {
      return BGIParseResult.error(ErrorCode.DATA_TOO_SHORT);
    }

    if (originalData[0] != 7) {
      return BGIParseResult.error(ErrorCode.GOBLINLESS);
    }

    buffer.position(1); // Skip initial 7

    try {
      int dataVersion = getShort();
      var properties = parseProperties();
      return BGIParseResult.success(new BGIData(dataVersion, properties));
    } catch (BGIParseException e) {
      return BGIParseResult.error(e);
    }
  }

  private Map<Integer, Object> parseProperties() throws BGIParseException {
    int numAttributes = getShort();

    var properties = new HashMap<Integer, Object>();
    for (int i = 0; i < numAttributes; i++) {
      int key = getInt();
      byte typeId = getByte();

      Object value = switch (typeId) {
        case 2 -> getVarInt();
        case 3 -> getString();
        case 4 -> getStringArray();
        case 8 -> getDouble();
        case 9 -> getStream();
        case 10 -> getBoolean();
        default -> throw new BGIParseException("Could not parse data type [" + typeId + "]", ErrorCode.UNKNOWN_DATA_TYPE);
      };

      properties.put(key, value);
    }

    return properties;
  }

  private byte getByte() {
    return buffer.get();
  }

  private short getShort() {
    return buffer.getShort();
  }

  private double getDouble() {
    return buffer.getDouble();
  }

  private boolean getBoolean() {
    return getByte() != 0;
  }

  private String getString() {
    int length = getVarInt();
    byte[] strBytes = new byte[length];
    buffer.get(strBytes);
    return new String(strBytes, StandardCharsets.UTF_8);
  }

  private String[] getStringArray() {
    int count = getVarInt();
    String[] array = new String[count];
    for (int j = 0; j < count; j++) {
      array[j] = getString();
    }
    return array;
  }

  private byte[] getStream() {
    int length = getVarInt();
    byte[] streamBytes = new byte[length];
    buffer.get(streamBytes);
    return streamBytes;
  }

  private int getInt() {
    return buffer.getInt();
  }

  private int getVarInt() {
    int value = 0;
    int shift = 0;

    while (buffer.hasRemaining()) {
      byte b = getByte();

      // Get the 7 data bits
      int temp = b & 0x7F;
      value |= temp << shift;

      // If the continuation bit is not set, we're done
      if ((b & 0x80) == 0) {
        break;
      }

      // Each byte contributes 7 bits
      shift += 7;

      // guard against malformed input
      if (shift > 32) {
        throw new IllegalArgumentException("Malformed variable int");
      }
    }

    return value;
  }

//  public static void main(String[] args) {
//    String item1Str = "070001000F8FA7BDAB03044550494394412F3A04011176616E697368696E675F63757273652031999609E9092B08454E4348414E5453010108454E4348414E545304011176616E697368696E675F637572736520310000009EF4F3FF02C801A8EEBDC7030B434F52455F484F52524944AF3300CB084014000000000000D0073349030B434F52455F484F52524944E5925CAC0307434F52455F54360023BEF604031826374974277320646973696E746567726174696E672E2E2E0011266344726F7073206F6E2064656174682E0024728B03173C746965722D636F6C6F723E486F7272696420436F7265002749E20840180000000000000C4703F8030AC2AB20436F726520C2BB283BC0E6030A434F4E53554D41424C45318E8960030A636F6E73756D61626C6566DC82FB0A01";
//    byte[] item1 = hexStringToByteArray(item1Str);
//    System.out.println(Arrays.toString(item1));
//    BGIParseResult<BGIData> result = BGIParser.parse(item1);
//    if (result.isError()) {
//      System.out.println(result.error().get().getMessage());
//    }
//  }
//
//  public static byte[] hexStringToByteArray(String hex) {
//    int len = hex.length();
//    if (len % 2 != 0) {
//      throw new IllegalArgumentException("Hex string must have even length");
//    }
//    byte[] result = new byte[len / 2];
//    for (int i = 0; i < len; i += 2) {
//      result[i / 2] = (byte) ((Character.digit(hex.charAt(i), 16) << 4)
//          + Character.digit(hex.charAt(i+1), 16));
//    }
//    return result;
//  }
}
