package dev.bnjc.bglib;

import dev.bnjc.bglib.exceptions.BGIParseException;
import dev.bnjc.bglib.exceptions.ErrorCode;
import java.nio.ByteBuffer;
import java.nio.ByteOrder;
import java.nio.charset.StandardCharsets;
import java.util.HashMap;
import java.util.Map;
import java.util.Optional;
import net.minecraft.class_1799;
import net.minecraft.class_2487;
import net.minecraft.class_2520;
import net.minecraft.class_9279;
import net.minecraft.class_9334;

public final class BGIParser {
  public static final String BGI_TAG = "bgi";

  private final ByteBuffer buffer;
  private final byte[] originalData;

  private BGIParser(byte[] data) {
    this.originalData = data;
    this.buffer = ByteBuffer.wrap(data);
    this.buffer.order(ByteOrder.BIG_ENDIAN);
  }

  public static BGIParseResult<BGIData> parse(byte[] data) {
    var parser = new BGIParser(data);
    return parser.parse();
  }

  public static BGIParseResult<BGIData> parse(class_1799 itemStack) {
    class_2487 stackNbt = itemStack.method_57825(class_9334.field_49628, class_9279.field_49302).method_57461();

    if (stackNbt != null && stackNbt.method_10573(BGI_TAG, class_2520.field_33257)) {
      return BGIParser.parse(stackNbt.method_10547(BGI_TAG));
    }

    return BGIParseResult.error(ErrorCode.MISSING_TAG);
  }

  private BGIParseResult<BGIData> parse() {
    if (originalData.length < 9) {
      return BGIParseResult.error(ErrorCode.DATA_TOO_SHORT);
    }

    if (originalData[0] != 7) {
      return BGIParseResult.error(ErrorCode.GOBLINLESS);
    }

    buffer.position(1); // Skip initial 7

    try {
      int dataVersion = getShort();
      var properties = parseProperties();
      return BGIParseResult.success(new BGIData(dataVersion, properties));
    } catch (BGIParseException e) {
      return BGIParseResult.error(e);
    }
  }

  private Map<Integer, Object> parseProperties() throws BGIParseException {
    int numAttributes = getShort();

    var properties = new HashMap<Integer, Object>();
    for (int i = 0; i < numAttributes; i++) {
      int key = getVarInt();
      byte typeId = getByte();

      Object value = switch (typeId) {
        case 2 -> getVarInt();
        case 3 -> getString();
        case 4 -> getStringArray();
        case 8 -> getDouble();
        case 9 -> getStream();
        case 10 -> getBoolean();
        default -> throw new BGIParseException("Could not parse data type [" + typeId + "]", ErrorCode.UNKNOWN_DATA_TYPE);
      };

      properties.put(key, value);
    }

    return properties;
  }

  private byte getByte() {
    return buffer.get();
  }

  private short getShort() {
    return buffer.getShort();
  }

  private double getDouble() {
    return buffer.getDouble();
  }

  private boolean getBoolean() {
    return getByte() != 0;
  }

  private String getString() {
    int length = getVarInt();
    byte[] strBytes = new byte[length];
    buffer.get(strBytes);
    return new String(strBytes, StandardCharsets.UTF_8);
  }

  private String[] getStringArray() {
    int count = getVarInt();
    String[] array = new String[count];
    for (int j = 0; j < count; j++) {
      array[j] = getString();
    }
    return array;
  }

  private byte[] getStream() {
    int length = getVarInt();
    byte[] streamBytes = new byte[length];
    buffer.get(streamBytes);
    return streamBytes;
  }

  private int getVarInt() {
    int value = 0;
    int shift = 0;

    while (buffer.hasRemaining()) {
      byte b = getByte();

      // Get the 7 data bits
      int temp = b & 0x7F;
      value |= temp << shift;

      // If the continuation bit is not set, we're done
      if ((b & 0x80) == 0) {
        break;
      }

      // Each byte contributes 7 bits
      shift += 7;

      // guard against malformed input
      if (shift > 32) {
        throw new IllegalArgumentException("Malformed varint");
      }
    }

    return value;
  }

  // @ExcludeSection:start
  public static void main(String[] args) {
//    byte[] input = {7,0,0,0,1,0,0,0,33,70,121,-42,-72,8,64,17,-118,87,-89,-122,-62,39,-122,-31,26,32,8,64,58,-80,104,-37,-117,-84,113,66,92,35,-98,3,0,0,0,21,123,34,83,116,97,116,34,58,34,80,86,69,95,68,65,77,65,71,69,34,125,0,1,64,66,3,0,0,0,10,83,69,84,95,87,73,90,65,82,68,0,39,73,-30,3,0,0,0,9,82,65,82,69,95,71,69,65,82,-98,-12,-13,-1,2,0,0,0,1,-82,-121,71,22,4,0,0,0,28,0,0,0,18,38,55,-62,-85,32,77,97,103,101,32,65,114,109,111,114,32,-62,-69,0,0,0,0,0,0,0,27,38,51,32,38,55,-30,-101,-88,32,68,101,102,101,110,115,101,58,32,38,102,38,50,43,50,54,46,55,0,0,0,0,0,0,0,36,38,51,32,38,55,-16,-97,-105,-95,32,67,114,105,116,105,99,97,108,32,80,111,119,101,114,58,32,38,102,38,50,43,52,46,51,57,37,0,0,0,32,38,51,32,38,55,-30,-102,-110,32,66,108,117,110,116,32,82,97,116,105,110,103,58,32,38,102,38,50,49,46,51,55,37,0,0,0,0,0,0,0,32,38,51,32,38,55,-16,-97,-105,-95,32,80,118,69,32,68,97,109,97,103,101,58,32,38,102,38,50,43,50,46,50,53,37,0,0,0,32,38,51,32,38,55,-30,-104,-124,32,77,97,103,105,99,32,68,97,109,97,103,101,58,32,38,102,38,50,43,54,46,52,37,0,0,0,38,38,51,32,38,55,-30,-104,-82,32,84,104,97,117,109,97,116,117,114,103,121,32,80,111,119,101,114,58,32,38,102,38,50,43,53,46,52,53,37,0,0,0,39,38,51,32,38,55,-16,-97,-113,-71,32,80,114,111,106,101,99,116,105,108,101,32,68,97,109,97,103,101,58,32,38,102,38,50,43,53,46,48,54,37,0,0,0,0,0,0,0,34,38,51,32,38,55,-30,-116,-102,32,77,111,118,101,109,101,110,116,32,83,112,101,101,100,58,32,38,102,38,50,43,48,46,52,37,0,0,0,0,0,0,0,32,38,51,32,38,55,-30,-104,-128,32,69,110,99,104,97,110,116,58,32,38,102,85,110,98,114,101,97,107,105,110,103,32,73,0,0,0,0,0,0,0,29,38,51,32,38,55,-31,-102,-79,32,-62,-89,52,86,105,111,108,101,110,99,101,32,45,32,82,97,110,107,32,52,0,0,0,37,38,51,32,38,55,-31,-102,-79,32,69,109,112,116,121,32,38,50,82,117,110,101,99,97,114,118,105,110,103,38,55,32,83,111,99,107,101,116,0,0,0,0,0,0,0,27,38,55,75,101,101,112,115,32,121,111,117,114,32,98,97,99,111,110,32,117,110,98,117,114,110,116,46,0,0,0,0,0,0,0,27,38,55,91,38,54,87,105,122,97,114,100,38,55,93,32,38,102,83,101,116,32,66,111,110,117,115,58,0,0,0,24,38,56,91,52,93,32,120,49,46,50,53,32,77,97,103,105,99,32,68,97,109,97,103,101,0,0,0,29,38,56,91,52,93,32,43,49,48,37,32,67,111,111,108,100,111,119,110,32,82,101,100,117,99,116,105,111,110,0,0,0,32,38,56,91,52,93,32,43,50,48,37,32,70,97,108,108,32,68,97,109,97,103,101,32,82,101,100,117,99,116,105,111,110,0,0,0,0,0,0,0,15,38,55,84,105,101,114,58,32,-62,-89,98,82,65,82,69,0,0,0,25,38,55,68,117,114,97,98,105,108,105,116,121,58,32,49,48,48,48,32,47,32,49,48,48,48,0,35,-66,-10,4,0,0,0,1,0,0,0,27,38,55,75,101,101,112,115,32,121,111,117,114,32,98,97,99,111,110,32,117,110,98,117,114,110,116,46,-7,-91,118,48,9,0,0,0,125,0,0,0,1,0,0,0,1,0,0,0,11,82,117,110,101,99,97,114,118,105,110,103,0,0,0,1,0,0,0,20,-62,-89,52,86,105,111,108,101,110,99,101,32,45,32,82,97,110,107,32,52,24,-74,111,-23,-30,80,75,40,-106,113,-90,-53,-115,-65,-102,19,1,0,0,0,21,82,85,78,69,67,65,82,86,73,78,71,95,79,70,70,69,78,83,69,95,52,1,0,0,0,11,82,85,78,69,67,65,82,86,73,78,71,0,0,0,0,11,82,117,110,101,99,97,114,118,105,110,103,4,-29,92,122,3,0,0,0,23,123,34,83,116,97,116,34,58,34,66,76,85,78,84,95,82,65,84,73,78,71,34,125,104,-98,-2,-123,8,64,20,56,-17,52,-42,-95,98,40,59,-64,-26,3,0,0,0,5,65,82,77,79,82,-51,-12,16,105,3,0,0,0,32,123,34,83,116,97,116,34,58,34,67,82,73,84,73,67,65,76,95,83,84,82,73,75,69,95,80,79,87,69,82,34,125,-100,110,-102,-49,4,0,0,0,1,0,0,0,8,38,54,72,101,97,116,101,100,-65,-23,113,75,8,63,-11,-24,-89,29,-26,-102,-44,121,-12,84,-106,10,1,73,37,-72,6,3,0,0,0,25,123,34,83,116,97,116,34,58,34,77,79,86,69,77,69,78,84,95,83,80,69,69,68,34,125,111,-126,98,97,8,64,25,-104,95,6,-10,-108,70,-11,-6,21,89,10,1,-86,83,-72,-74,3,0,0,0,28,123,34,83,116,97,116,34,58,34,80,82,79,74,69,67,84,73,76,69,95,68,65,77,65,71,69,34,125,-84,-126,1,-28,2,0,0,3,-24,-88,-18,-67,-57,3,0,0,0,24,66,76,79,79,68,83,78,79,85,84,95,77,65,71,73,95,76,69,71,71,73,78,71,83,90,-7,-36,-105,8,63,58,54,-30,-21,28,67,45,74,-103,-49,-74,3,0,0,0,29,123,34,83,116,97,116,34,58,34,84,72,65,85,77,65,84,85,82,71,89,95,68,65,77,65,71,69,34,125,0,36,114,-117,3,0,0,0,34,60,116,105,101,114,45,99,111,108,111,114,62,66,108,111,111,100,115,110,111,117,116,32,77,97,103,105,115,32,80,97,110,116,115,85,-73,67,-57,8,64,21,-47,-100,-32,117,-10,-3,114,-75,-86,-122,10,1,16,-73,70,-64,10,1,-108,65,47,58,4,0,0,0,2,0,0,0,12,117,110,98,114,101,97,107,105,110,103,32,49,0,0,0,10,105,110,102,105,110,105,116,121,32,49,62,-45,56,-81,8,64,1,-10,-3,33,-1,46,73,-27,-110,92,-84,3,0,0,0,6,65,82,67,65,78,69,12,71,3,-8,3,0,0,0,16,-62,-85,32,77,97,103,101,32,65,114,109,111,114,32,-62,-69,-103,-106,9,-23,3,0,0,0,19,123,34,83,116,97,116,34,58,34,69,78,67,72,65,78,84,83,34,125};
    byte[] input = {7,0,0,0,1,0,0,0,19,121,-12,84,-106,10,1,-11,-6,21,89,10,1,51,-29,8,-64,8,63,-32,0,0,0,0,0,0,0,39,73,-30,3,0,0,0,5,81,85,69,83,84,102,-36,-126,-5,10,1,-88,-18,-67,-57,3,0,0,0,22,83,84,65,82,84,69,82,95,76,85,77,66,69,82,74,65,67,75,95,79,65,75,51,-30,-15,-120,8,63,-16,0,0,0,0,0,0,-98,-12,-13,-1,2,0,0,0,1,49,-114,-119,96,3,0,0,0,10,99,111,110,115,117,109,97,98,108,101,0,36,114,-117,3,0,0,0,30,60,116,105,101,114,45,99,111,108,111,114,62,67,97,114,118,101,100,32,83,116,117,109,112,32,45,32,79,97,107,-82,-121,71,22,4,0,0,0,13,0,0,0,18,38,55,-62,-85,32,81,117,101,115,116,32,73,116,101,109,32,-62,-69,0,0,0,0,0,0,0,31,85,115,101,32,116,104,105,115,32,105,116,101,109,32,116,111,32,115,116,97,114,116,32,97,32,113,117,101,115,116,33,0,0,0,36,89,79,85,32,67,65,78,32,79,78,76,89,32,68,79,32,79,78,69,32,81,85,69,83,84,32,65,84,32,65,32,84,73,77,69,33,0,0,0,41,85,83,73,78,71,32,84,72,73,83,32,87,73,76,76,32,82,69,77,79,86,69,32,89,79,85,82,32,65,67,84,73,86,69,32,81,85,69,83,84,33,0,0,0,0,0,0,0,28,38,55,73,116,39,115,32,99,111,118,101,114,101,100,32,105,110,32,100,101,101,112,32,99,117,116,115,46,0,0,0,32,38,55,84,104,101,121,32,107,105,110,100,32,111,102,32,114,101,115,101,109,98,108,101,32,119,111,114,100,115,46,46,46,0,0,0,0,0,0,0,21,38,55,82,101,112,108,97,110,116,105,110,103,32,80,114,111,106,101,99,116,58,0,0,0,16,38,55,56,32,79,97,107,32,83,97,112,108,105,110,103,115,0,0,0,0,0,0,0,16,38,55,84,105,101,114,58,32,-62,-89,54,81,85,69,83,84,114,-75,-86,-122,10,1,10,-69,-6,-120, 9,0,0,0,59,0,0,0,1,0,0,0,1,0,0,0,37,109,109,111,99,111,114,101,32,113,117,101,115,116,32,115,116,97,114,116,32,37,112,108,97,121,101,114,37,32,112,97,117,108,45,111,97,107,0,0,0,0,0,0,0,0,1,0,0,35,-66,-10,4,0,0,0,9,0,0,0,31,85,115,101,32,116,104,105,115,32,105,116,101,109,32,116,111,32,115,116,97,114,116,32,97,32,113,117,101,115,116,33,0,0,0,36,89,79,85,32,67,65,78,32,79,78,76,89,32,68,79,32,79,78,69,32,81,85,69,83,84,32,65,84,32,65,32,84,73,77,69,33,0,0,0,41,85,83,73,78,71,32,84,72,73,83,32,87,73,76,76,32,82,69,77,79,86,69,32,89,79,85,82,32,65,67,84,73,86,69,32,81,85,69,83,84,33,0,0,0,0,0,0,0,28,38,55,73,116,39,115,32,99,111,118,101,114,101,100,32,105,110,32,100,101,101,112,32,99,117,116,115,46,0,0,0,32,38,55,84,104,101,121,32,107,105,110,100,32,111,102,32,114,101,115,101,109,98,108,101,32,119,111,114,100,115,46,46,46,0,0,0,0,0,0,0,21,38,55,82,101,112,108,97,110,116,105,110,103,32,80,114,111,106,101,99,116,58,0,0,0,16,38,55,56,32,79,97,107,32,83,97,112,108,105,110,103,115,16,-73,70,-64,10,1,83,52,-46,-52,3,0,0,0,21,69,78,84,73,84,89,95,80,76,65,89,69,82,95,76,69,86,69,76,85,80,-108,65,47,58,4,0,0,0,1,0,0,0,12,117,110,98,114,101,97,107,105,110,103,32,49,40,59,-64,-26,3,0,0,0,5,81,85,69,83,84,-103,-106,9,-23,3,0,0,0,19,123,34,83,116,97,116,34,58,34,69,78,67,72,65,78,84,83,34,125};

    var data = BGIParser.parse(input);
    data.ifSuccess(item -> {
      System.out.println(item);

      Optional<String> itemId = item.getString(BGIField.ITEM_ID);
      itemId.ifPresent(System.out::println);
    });
  }
  // @ExcludeSection:end
}
